/** @file   bmfont.h
 * @brief   Declaraction of BMFont - class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_BMFONT_H
#define H_ENG2D_BMFONT_H

#include "eng2d_dll.h"
#include <string>
#include <map>
#include "glyphset.h"
#include "eng2d_Exception.h"


namespace eng2d {


/** @class  BMFont
 * @brief   Represents a hightcolor Font with several different sized GlyphSets.
 * @author  Tomi Lamminsaari
 *
 * A single @c BMFont can have several different charactersets. They can be
 * different sized or even look very different.
 */
class DLLIMPORT BMFont
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================


  ///
  /// Constructors, destructor and operators
  /// ======================================
  
  /** Constructs new font
   */
  BMFont();

	/** Constructor.
   * @param     fontname        The fontfile we want to load.
   * @exception xIoError        Thrown if there are problems with font file.
   */
	BMFont( const std::string& fontname ) throw (xIoError);


	/** Destructor
   */
	virtual ~BMFont();


private:

	/** Copy constructor.
   * @param     rO                Reference to another BMFont
   */
  BMFont( const BMFont& rO );

	/** Assignment operator
   * @param     rO                Reference to another BMFont
   * @return    Reference to us.
   */
  BMFont& operator = ( const BMFont& rO );

public:


  ///
  /// Methods
  /// =======

  /** Loads new data to this font. The previus fontdata will be destroyed.
   * @param     fontname          Name of the fontfile
   * @return    An errorcode.
   * @retval    KErrNone          Loading was successful.
   * @retval    KErrNotFound      Fontfile was not found.
   * @retval    KErrEof           EOF encountered while parsing font data.
   */
  int open( const std::string& fontname );
  
  /** Closes this font.
   */
  void close();
  


  ///
  /// Getter methods
  /// ==============

  /** Tells if this font has data for given textsize.
   * @param     s                 Requested size
   * @return    <code>true</code> if there is data for requested fontsize.
   */
  bool hasSize( int s ) const;

  /** Returns the glyphset that holds the data for given sized font.
   * @param     s                 Requested size
   * @return    Pointer to glyphset or null-pointer if there is no
   *            matching size.
   */
  const GlyphSet* glyphset( int s ) const;
  
  /** Returns the glyphset with given id.
   * @param     id                The id of the glyphset
   * @return    Pointer to glyphset or null-pointer if there is no glyphset
   *            with given id.
   */
  const GlyphSet* glyphset( const std::string& id ) const;
  
  

protected:

  ///
  /// Members
  /// =======

  /** Name of this font */
  std::string               m_name;
  
  /** This map contains all the charactersets that form this font. The ID of
   * the charactersets operates as the key to this map.
   */
  std::map<std::string, GlyphSet*>  m_glyphsets;
  


private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif
